function [lut, lutParams]= remap2di(lutXFilename, lutYFilename, m,n)
%
% Prepare / initialise a 2D lookup table for image remapping.
% Read information from files.
%
% Input params:
%
% lutXFilename, lutYFilename - (strings) refer to files
%  that contain cols/lins indexes to be combined at the "to return lut".
%  Alternatively the LUT matrices may be given directly.
%
% m, n - size of images to be remapped (m=inpImgNLins, n=inpImgNCols).
%  Returned lut will be valid ONLY for input images with size [m n]
%  (m is really the critical parameter).
%  This parameters are facultative. If not specified, size(lutX)
%  will be used.
%
% Output params:
%
% lut - 2D lookup table (really a vector); works only for images with
%   size == [m n]
%
% lutParams - parameters required to understand lut
%   lutParams= [outpImgNLins, outpImgNCols, ...
%               inpImgNLins,  inpImgNCols, minX,maxX, minY,maxY]
%   1<=minX<=maxX<=inpImgNCols, 1<=minY<=maxY<=inpImgNLins
%   
% See also: remap2d, remap2ds (save lut2d), remap2dc (change lut2d)

% 1/10/98, 27/5/01 (luts as params), J. Gaspar
%

if ~isstr(lutXFilename) | ~isstr(lutYFilename),
   %
   %error('lutXFilename or lutYFilename is not a string');
   %
   if isstr(lutXFilename) | isstr(lutYFilename),
      error('lutXFilename and lutYFilename not simultaneously strings or matrices');
   end
   lutX= lutXFilename;
   lutY= lutYFilename;
else
   lutX= pi5read(lutXFilename);
   lutY= pi5read(lutYFilename);
end

m2= size(lutX,1);
n2= size(lutX,2);

minX= min(min(lutX));
maxX= max(max(lutX));
minY= min(min(lutY));
maxY= max(max(lutY));

if nargin<4,
  if nargin<3,
    m= size(lutX,1);
  end
  n= size(lutX,2);
end

if minX<0 | minY<0 | maxX>n-1 | maxY>m-1,
  fprintf(1,'* minX=%f minY=%f \n* maxX=%f inpImgNCols=%f maxY=%f inpImgNLins=%f\n',[minX minY maxX n maxY m]);
  error('lutParams or inpImgSz problem: minX<0 | minY<0 | maxX>inpImgNCols-1 | maxY>inpImgNLins-1');
end

lutParams= [m2 n2 m n minX maxX minY maxY];

lut= reshape(lutX,m2*n2,1)*m + reshape(lutY,m2*n2,1) +1;
