#ifndef __CDK13_EMITTER_POSTFIX_H__
#define __CDK13_EMITTER_POSTFIX_H__

#include <iostream>
#include <string>
#include <memory>
#include <cdk/compiler.h>

namespace cdk {

  /**
   * The postfix code emitter defines an interface to be used by semantic
   * analysers, as defined by the strategy design pattern. Specific
   * implementations will provide the realization of the postfix commands for a
   * particular target machine.
   *
   * <h3>Rotation and shift instructions</h3>
   * Shift and rotation operations have as maximum value the number of bits of the
   * underlying processor register (32 bits in a ix86-family processor). Safe
   * operation for values above is not guaranteed.
   *
   * These operations use two values from the stack: the value at the top specifies
   * the number of bits to rotate/shift; the second from the top is the value to be
   * rotated/shifted.
   *
   * <h3>Logical instructions</h3>
   *
   * Logical instructions perform logical operations using the elements at the
   * top of the stack. Arguments are taken from the stack, the result is put on the
   * stack.
   *
   * <h3>Type conversion instructions</h3>
   *
   * Type conversion instructions perform elementary type conversions. The conversions
   * are from and to integers and simple and double precision floating point values.
   *
   * <h3>Integer comparison instructions</h3>
   *
   * The comparison instructions are binary operations that leave at the top of the
   * stack 0 (zero) or 1 (one), depending on the result result of the comparison:
   * respectively, <tt>false</tt> or <tt>true</tt>. The value may be directly used
   * to perform conditional jumps (e.g., JZ, JNZ), that use the value of the top of
   * the stack instead of relying on special processor registers (<i>flags</i>).
   *
   * <h3>Function definition instructions</h3>
   *
   * In a stack machine the arguments for a function call are already in the stack.
   * Thus, it is not necessary to put them there (it is enough not to remove them).
   * When building functions that conform to the C calling convetions,
   * those arguments are destroyed by the caller, <i>after</i> the return of the
   * callee, using <tt>TRASH</tt>, stating the total size (i.e., for all
   * arguments). Regarding the callee, it must create a distinct activation
   * register <tt>ENTER</tt> or <tt>START</tt>) and, when no longer
   * needed, destroy it (<tt>LEAVE</tt>). The latter action must be
   * performed immediately before returning control to the caller.
   *
   * Similarly, to return values from a function, the callee must call
   * <tt>POP</tt> to store the return value in the accumulator register, so
   * that it survives the destruction of the invocation context. The caller must
   * call <tt>PUSH</tt>, to put the accumulator in the stack. An analogous
   * procedure is valid for <tt>DPOP/DPUSH</tt> (for double precision floating point return
   * values).
   *
   * <h3>Addressing instructions</h3>
   *
   * Note [*4*] that these operations (ADDR, LOCAL) put at the top of the stack the
   * symbol's address, independently of its origin. O endereço pode posteriormente
   * ser utilizado como ponteiro, obtido o valor nesse endereço (LOAD) ou guardar
   * um valor nesse endereço (STORE). No entanto, nas duas últimas situações,
   * devido à frequência com que ocorrem e o número de ciclos de relógio que levam
   * a executar, podem ser substituídas com vantagem pela operações descritas em
   * [*10*].
   *
   * "Quick opcodes" are shortcuts for groups of operations commonly used
   * together. These opcodes may be made efficient by implementing them in
   * different ways than the original set of high-level operations would suggest,
   * i.e., the code generated by <tt>ADDRV</tt> may be more efficient than
   * the code generated by <tt>ADDR</tt> followed by <tt>LOAD</tt>.
   * Nevertheless, the outcome is the same.
   *
   * <h3>Load instructions</h3>
   *
   * The load instructions assume that the top of the stack contains an address
   * pointing to the data to be read. Each load instruction will replace the
   * address at the top of the stack with the contents of the position it points
   * to. Load instructions differ only in what they load.
   *
   * <h3>Store instructions</h3>
   *
   * Store instructions assume the stack contains at the top the address where
   * data is to be stored. That data is in the stack, immediately after (below) the
   * address. Store instructions differ only in what they store.
   *
   * <h3>Labels</h3>
   *
   * In a declaration of a symbol common to more than one module, other modules may
   * also contain common or external declarations. Nevertheless, only one
   * initialized declaration is allowed. Declarations need not be associated with
   * any particular segments.
   *
   * In a declaration common to several modules, any number of modules may
   * contain common or external declarations, but only one of them may
   * contain an initialized declaration. A declaration does not need to be
   * specified in a specific segment.
   */
  class basic_postfix_emitter {

    std::shared_ptr<compiler> &_compiler;

  protected:

    basic_postfix_emitter(std::shared_ptr<compiler> &compiler) :
        _compiler(compiler) {
    }

    std::ostream &os() {
      return *_compiler->ostream();
    }

    bool debug() {
      return _compiler->debug();
    }

  public:
    /**
     * Destructor: the only action is to flush the output stream.
     */
    virtual ~basic_postfix_emitter() {
      os().flush();
      _compiler = nullptr;
    }

  public:

    // Segment selection

    /**
     * Starts the data segment for uninitialized values.
     */
    virtual void BSS() = 0;

    /**
     * Starts the data segment for initialized values.
     */
    virtual void DATA() = 0;

    /**
     * Starts the data segment for initialized constant values.
     */
    virtual void RODATA() = 0;

    /**
     * Starts the text (code) segment.
     */
    virtual void TEXT() = 0;

  public:

    // Values (declaration in segments)

    /**
     * Declares a static byte.
     *
     * @param value is the byte to be declared.
     */
    virtual void SBYTE(char value) = 0;

    /**
     * Declares a static word.
     *
     * @param value is the word to be declared.
     */
    virtual void SSHORT(short value) = 0;

    /**
     * Declares a static integer value.
     *
     * @param value is the number to be declared.
     */
    virtual void SINT(int value) = 0;

    /**
     * Declares a static simple precision floating point value.
     *
     * @param value is the number to be declared.
     */
    virtual void SFLOAT(float value) = 0;

    /**
     * Declares a static double precision floating point value.
     *
     * @param value is the number to be declared.
     */
    virtual void SDOUBLE(double value) = 0;

    /**
     * Declares a static character string (no special characters are allowed in <tt>value</tt>).
     *
     * @param value is the string to be declared.
     */
    virtual void SSTRING(std::string value) = 0;

    /**
     * Declares an uninitialized vector with the length (in bytes) given as argument.
     *
     * @param value is the vector's length.
     */
    virtual void SALLOC(int value) = 0;

    /**
     * Declares a name for an address.
     *
     * @param label is the name to be declared.
     */
    virtual void SADDR(std::string label) = 0;

  public:

    // labels

    /**
     * Forces the alignment of code or data.
     */
    virtual void ALIGN() = 0;

    /**
     * Generates a new label, as indicated by the argument.
     *
     * @param label is the new label's name.
     */
    virtual void LABEL(std::string label) = 0;

    /**
     * Declares the symbol whose name is passed as argument as being externally defined, i.e., defined in another compilation module.
     *
     * @param label is the symbol's name.
     */
    virtual void EXTERN(std::string label) = 0;

    /**
     * Declare a name/label (first argument) with a given type (second argument; see below).
     * Declaration of a name must preceed its definition.
     *
     * @param label is the name to be defined.
     * @param type is the named object's type.
     * @see  FUNC OBJ NONE
     */
    virtual void GLOBAL(const char *label, std::string type) = 0;

    /**
     * Declare a name/label (first argument) with a given type (second argument; see below).
     * Declaration of a name must preceed its definition. This is the C++-friendly interface.
     *
     * @param label is the name to be defined.
     * @param type is the named object's type.
     * @see  FUNC OBJ NONE
     */
    virtual void GLOBAL(std::string label, std::string type) = 0;

  public:

    // Addressing, Loading and Storing

    /**
     * Absolute addressing instruction: puts the address of the name passed as argument at
     * the top of the stack.
     *
     * Absolute addressing <tt>ADDR</tt>) is performed using labels. Relative
     * addressing (<tt>LOCAL</tt>) requires a frame pointer to work: the frame
     * pointer defines an addressing reference.
     *
     * @param label is the name of the memory position.
     * @see LOCAL
     */
    virtual void ADDR(std::string label) = 0;

    /**
     * Absolute addressing instruction: quick opcode: the same as the sequence ADDR(name); STINT();
     *
     * @param label is the name of the memory position.
     */
    virtual void ADDRA(std::string label) = 0;

    /**
     * Absolute addressing instruction: quick opcode: the same as the sequence ADDR(name); LDINT();
     *
     * @param label is the name of the memory position.
     */
    virtual void ADDRV(std::string label) = 0;

    /**
     * Relative addressing instruction: puts at the top of the stack
     * the address of the local variable, obtained by computing the
     * offset relative to the frame pointer.
     *
     * The value passed as argument is as follows: greater of equal to
     * 8, means function arguments; equal to 4, means the function's
     * return address; equal to 0, means the frame pointer itself; less
     * that -4, means local variables.
     *
     * @param offset from the top of the stack.
     */
    virtual void LOCAL(int offset) = 0;

    /**
     * Relative addressing instruction: quick opcode: the same as the sequence LOCAL(offset); STINT();
     *
     * @param offset from the top of the stack.
     */
    virtual void LOCA(int offset) = 0;

    /**
     * Relative addressing instruction: quick opcode: the same as the sequence ADDR(name); LDINT();
     *
     * @param offset from the top of the stack.
     */
    virtual void LOCV(int offset) = 0;

  public:

    // Load operations

    /**
     * Load instruction: loads 1 byte (char).
     */
    virtual void LDBYTE() = 0;

    /**
     * Load instruction: loads 2 bytes (short).
     */
    virtual void LDSHORT() = 0;

    /**
     * Load instruction: loads 4 bytes (integer -- rvalue).
     */
    virtual void LDINT() = 0;

    /**
     * Load instruction: loads a single precision floating point value.
     */
    virtual void LDFLOAT() = 0;

    /**
     * Load instruction: loads a double precision floating point value.
     */
    virtual void LDDOUBLE() = 0;

  public:

    // Store operations

    /**
     * Store instruction: stores 1 byte (char).
     */
    virtual void STBYTE() = 0;

    /**
     * Store instruction: stores 2 bytes (short).
     */
    virtual void STSHORT() = 0;

    /**
     * Store instruction: stores 4 bytes (integer).
     */
    virtual void STINT() = 0;

    /**
     * Store instruction: stores a single precision floating point value.
     */
    virtual void STFLOAT() = 0;

    /**
     * Store instruction: stores a double precision floating point value.
     */
    virtual void STDOUBLE() = 0;

  public:

    // Simple Stack Operations

    /**
     * Pushes to the stack the value of the stack pointer.
     */
    virtual void SP() = 0;

    /**
     * Allocates in the stack as many bytes as indicated by the value at the top of the stack.
     *
     * Dynamic memory allocation in the stack, equivalent to a call to the
     * C language <tt>alloca</tt> function, changes the offsets of temporary
     * variables that may exist in the stack when the allocation is performed. Thus,
     * it should only be used when no temporary variables exist, or when the full
     * import of its actions is fully understood.
     */
    virtual void ALLOC() = 0;

    /**
     * Duplicates the value at the top of the stack.
     */
    virtual void DUP32() = 0;

    /**
     * Duplicates the double precision value at the top of the stack.
     */
    virtual void DUP64() = 0;

    /**
     * Exchanges the two elements at the top of the stack.
     */
    virtual void SWAP32() = 0;

    /**
     * Exchanges the two elements at the top of the stack.
     */
    virtual void SWAP64() = 0;

    /**
     * Pushes a integer value to the stack top.
     *
     * @param value is the value to be put in the stack.
     */
    virtual void INT(int value) = 0;

    /**
     * Pushes a single precision floating point value to the stack top.
     *
     * @param value is the value to be put in the stack.
     */
    virtual void FLOAT(float value) = 0;

    /**
     * Pushes a double precision floating point value to the stack top.
     *
     * @param value is the value to be put in the stack.
     */
    virtual void DOUBLE(double value) = 0;

  public:

    // Integer operations

    /**
     * Arithmetic instruction: negation (symmetric) of integer value.
     */
    virtual void NEG() = 0;

    /**
     * Arithmetic instruction: integer sum of two integer values.
     */
    virtual void ADD() = 0;

    /**
     * Arithmetic instruction: integer subtraction of two integer values.
     */
    virtual void SUB() = 0;

    /**
     * Arithmetic instruction: integer multiplication of two integer values.
     */
    virtual void MUL() = 0;

    /**
     * Arithmetic instruction: integer division of two integer values.
     */
    virtual void DIV() = 0;

    /**
     * Arithmetic instruction: integer division of two natural (integer) values.
     */
    virtual void UDIV() = 0;

    /**
     * Arithmetic instruction: remainder of integer division of two integer values.
     */
    virtual void MOD() = 0;

    /**
     * Arithmetic instruction: remainder of integer division of two natural (integer) values.
     */
    virtual void UMOD() = 0;

  public:

    // Floating-point operations

    /**
     * Arithmetic instruction: floating point negation of double value (symmetric).
     */
    virtual void DNEG() = 0;

    /**
     * Arithmetic instruction: floating point sum of two double values.
     */
    virtual void DADD() = 0;

    /**
     * Arithmetic instruction: floating point subtraction of two double values.
     */
    virtual void DSUB() = 0;

    /**
     * Arithmetic instruction: floating point multiplication of two double values.
     */
    virtual void DMUL() = 0;

    /**
     * Arithmetic instruction: floating point division of two double values.
     */
    virtual void DDIV() = 0;

  public:

    // Increment and Decrement Operations

    /**
     * Adds <tt>value</tt> to the value at the position defined by the address at
     * the top of the stack, i.e. <tt>[a]</tt> becomes <tt>[a]+value</tt>.
     *
     * @param value is the value to be added.
     */
    virtual void INCR(int value) = 0;

    /**
     * Subtracts <tt>value</tt> from the value at the position defined by the
     * address at the top of the stack, i.e. <tt>[a]</tt> becomes <tt>[a]-value</tt>.
     *
     * @param value is the value to be subtracted.
     */
    virtual void DECR(int value) = 0;

  public:

    // Type Conversion Operations

    /**
     * Type conversion instructions: converts from double precision floating point to simple precision floating point.
     */
    virtual void D2F() = 0;

    /**
     * Type conversion instructions: converts from simple precision floating point to double precision floating point.
     */
    virtual void F2D() = 0;

    /**
     * Type conversion instructions: converts from double precision floating point to integer.
     */
    virtual void D2I() = 0;

    /**
     * Type conversion instructions: converts from integer to double precision floating point.
     */
    virtual void I2D() = 0;

  public:

    // Comparison instructions

    /**
     * Integer comparison instructions: <i>equal to</i>.
     */
    virtual void EQ() = 0;

    /**
     * Integer comparison instructions: <i>not equal to</i>.
     */
    virtual void NE() = 0;

    /**
     * Integer comparison instructions: <i>greater than</i>.
     */
    virtual void GT() = 0;

    /**
     * Integer comparison instructions: <i>greater than or equal to</i>.
     */
    virtual void GE() = 0;

    /**
     * Integer comparison instructions: <i>less than or equal to</i>.
     */
    virtual void LE() = 0;

    /**
     * Integer comparison instructions: <i>less than</i>.
     */
    virtual void LT() = 0;

    /**
     * Integer comparison instructions: <i>greater than</i> for natural numbers (unsigned integers).
     */
    virtual void UGT() = 0;

    /**
     * Integer comparison instructions: <i>greater than or equal to</i> for natural numbers (unsigned integers).
     */
    virtual void UGE() = 0;

    /**
     * Integer comparison instructions: <i>less than or equal to</i> for natural numbers (unsigned integers).
     */
    virtual void ULE() = 0;

    /**
     * Integer comparison instructions: <i>less than</i> for natural numbers (unsigned integers).
     */
    virtual void ULT() = 0;

    /**
     * Compares two double precision floating point values. The result is an
     * integer value: less than 0, if the <tt>d1</tt> is less than <tt>d2</tt>;
     * 0, if they are equal; greater than 0, otherwise.
     */
    virtual void DCMP() = 0;

  public:

    // Bitwise Operations

    /**
     * Logical instructions: logical negation (bitwise), i.e., one's complement.
     */
    virtual void NOT() = 0;

    /**
     * Logical instructions: logical (bitwise) AND operation.
     */
    virtual void AND() = 0;

    /**
     * Logical instructions: logical (bitwise) OR operation.
     */
    virtual void OR() = 0;

    /**
     * Logical instructions: logical (bitwise) XOR operation.
     */
    virtual void XOR() = 0;

  public:

    // Rotation and Shift Operations

    /**
     * Rotation and shift instructions: left rotation of stack top.
     */
    virtual void ROTL() = 0;

    /**
     * Rotation and shift instructions: right rotation of stack top.
     */
    virtual void ROTR() = 0;

    /**
     * Rotation and shift instructions: left shift of stack top.
     */
    virtual void SHTL() = 0;

    /**
     * Rotation and shift instructions: right shift of stack top (unsigned).
     */
    virtual void SHTRU() = 0;

    /**
     * Rotation and shift instructions: right shift of stack top (signed).
     */
    virtual void SHTRS() = 0;

  public:

    // Starting a function

    /**
     * Function definition instructions: starts a function: push the frame pointer
     * (activation register) to the stack and allocate space for local variables,
     * according to the size given as argument (in bytes).
     *
     * @param bytes is the space to be allocated for local variables.
     */
    virtual void ENTER(size_t bytes) = 0;

    /**
     * Function definition instructions: equivalent to <tt>ENTER(0)</tt>.
     */
    virtual void START() = 0;

  public:

    // Leaving a function

    /**
     * Function definition instructions: removes a value from the stack (to the accumulator register).
     */
    virtual void STFVAL32() = 0;

    /**
     * Function definition instructions: removes a double precision floating point value from the stack (to a double prevision floating point register).
     */
    virtual void STFVAL64() = 0;

    /**
     * Function definition instructions: ends a function: restores the frame pointer
     * (activation register) and destroys the function-local stack data.
     */
    virtual void LEAVE() = 0;

    /**
     * Function definition instructions: returns from a function (the stack should contain the return address).
     */
    virtual void RET() = 0;

    /**
     * Function definition instructions: returns from a function, but removes
     * <tt>n</tt> bytes from the stack after removing the return address.
     * More or less the same as <tt>RET()</tt> followed by <tt>TRASH(n)</tt>.
     *
     * @param bytes the number of bytes to take from the stack.
     */
    virtual void RETN(int bytes) = 0;

  public:

    // Function calls

    /**
     * Calls the named function. Stores the return address in the stack.
     *
     * @param label is the called function's name.
     */
    virtual void CALL(std::string label) = 0;

    /**
     * Function definition instructions: removes <tt>n</tt> bytes from the stack.
     *
     * @param bytes the number of bytes to take from the stack.
     */
    virtual void TRASH(int bytes) = 0;

    /**
     * Function definition instructions: pushes the value in the accumulator register to the stack.
     */
    virtual void LDFVAL32() = 0;

    /**
     * Function definition instructions: pushes the value in the double precision floating point register to the stack.
     */
    virtual void LDFVAL64() = 0;

  public:

    // Basic Jump Operations

    virtual void JMP(std::string) = 0;
    virtual void LEAP() = 0;
    virtual void BRANCH() = 0;

  public:

    // Conditional Jump Operations[edit]

    virtual void JZ(std::string) = 0;
    virtual void JNZ(std::string) = 0;

    virtual void JEQ(std::string) = 0;
    virtual void JNE(std::string) = 0;

    virtual void JGT(std::string) = 0;
    virtual void JGE(std::string) = 0;
    virtual void JLE(std::string) = 0;
    virtual void JLT(std::string) = 0;

    virtual void JUGT(std::string) = 0;
    virtual void JUGE(std::string) = 0;
    virtual void JULE(std::string) = 0;
    virtual void JULT(std::string) = 0;

  public:

    // miscellaneous

    virtual void NIL() = 0;
    virtual void NOP() = 0;

  public:
    /* The following are used as the 2nd argument of GLOBL */

    /* unknown global label type */
    virtual std::string NONE() const {
      return "";
    }

    /* global label is a function */
    virtual std::string FUNC() const {
      return ":function";
    }

    /* global label is data */
    virtual std::string OBJ() const {
      return ":object";
    }

  };

} // cdk

#endif
