function myexp(cmd, options)
%
% A user interface to help doing Bode diagrams
%

% 12.5.09 J. Gaspar

if nargin<1
   cmd= 'who';
end
if nargin<2
   options= [];
end

% convert a numeric argument to a field in a struct
%
if isnumeric(options)
   options= struct('arg', options);
end

% global variables to hold all the data
%
global MYEXP MYEXPD
if isempty(MYEXP)
   MYEXP= struct('id', 1, ... % interf parameters
      'simDC',[], 'simAmpl',[], 'simFreqHz',[]);
   MYEXPD= { [] }; % simulation results
   for i=1:9
      myexp('new');
   end
   myexp('set', 1);
end

% the main commands
%
switch cmd
   case 'clear_all'
      MYEXP=[];
      % the next run of myexp.m will fill the data
      return
      
   case {'new', 'set', 'next', 'prev'}
      % set the current experiment id, or create a new one
      id= myexp_id(cmd, options);
      fprintf(1, '-- Current experiment id = %d.\n', id);
      
   case 'who'
      % list the data of the current experiment
      id= myexp_id('curr');
      if isfield(options, 'arg') && ~isempty(options.arg)
         id= options.arg;
      end
      show_myexp_info(id, options)

   case {'whos', 'whos_all'}
      % list all experiments
      id0= myexp_id('curr');
      options2= options;
      for id= 1:length(MYEXPD)
         if strcmp(cmd, 'whos')
            options2.min_info= [];
            show_myexp_info(id, options2)
         else
            options2.arg= id;
            myexp('who', options2);
         end
      end
      myexp('set', id0); % recover the original id
      
   case 'multi_freq'
      % input multiple frequencies
      if ~isfield(options, 'arg')
         disp('** warning undefined frequency list.')
         return
      end
      freqRad= options.arg;
      if prod(size(freqRad))==length(freqRad)
         % is a vector instead of Nx2
         w= [];
         for i=1:length(freqRad)
            w= [w; i freqRad(i)];
         end
         freqRad= w;
      end
      for i=1:size(w,1)
         myexp('set', w(i,1));
         myexp('mksrc', w(i,2));
         disp('** Please run the simulation and press return...');
         pause
         myexp('ld');
      end
      
   case {'mk_voltage_source', 'mksrc'}
      % create a sinewave VSTIM for the PSPICE simulation
      freqHz= 1e3/(2*pi); % 1krad/sec
      if ~isfield(options, 'arg')
         disp('** warning undefined frequency')
      else
         freqHz= options.arg/(2*pi); % arg comes in rad/sec
      end
      sim_config_data_write(freqHz);
      fprintf(1, '-- configured sinewave freq to:\n');
      fprintf(1, '\t%f [rad/sec] \t%f [Hz]\n', 2*pi*freqHz, freqHz);
      
   case {'load_pspice_result', 'loadres', 'load', 'ld'}
      % load the simulation results obtained with PSPICE
      % first verify that the simulation has a date after the stimulus file
      while 1
         d1= dir('dbode.stl');
         d2= dir('dbode-schematic1-simulacao_dc.dat');
         if d2.datenum > d1.datenum
            break
         else
            fprintf(1, '** WARNING date of results file OLDER than\n');
            fprintf(1, '   the date of the stimulus file\n');
            fprintf(1, '   PLEASE re-run the simulation\n');
            fprintf(1, '   or press i to ignore...');
            s= input('', 's');
            if strcmp(s, 'i')
               break
            end
         end
      end
      % finaly the load...
      id= myexp_id('curr');
      MYEXPD{id}= readdat('dbode-schematic1-simulacao_dc.dat');
      disp(['-- Loaded: dbode-schematic1-simulacao_dc.dat'])
      ret= sim_config_data_read;
      MYEXP.simDC(id)= ret(1);
      MYEXP.simAmpl(id)= ret(2);
      MYEXP.simFreqHz(id)= ret(3);

   case {'show_signal', 'sh'}
      % display signals Vin, Vout1..Vout7
      id= myexp_id('curr');
      if isfield(options, 'arg') && ~isempty(options.arg)
         options.sid= options.arg;
      end
      show_signal(id, options);
      
   case 'anim'
      % show the desired output signals for a list of freqs
      %  first parse the args
      %
      sid= 0:6;
      if isfield(options, 'arg') && ~isempty(options.arg)
         sid= options.arg;
      end
      if isfield(options, 'sid')
         sid= options.sid;
      end
      anim_list= 1:length(MYEXPD);
      if isfield(options, 'anim_list')
         anim_list= options.anim_list;
      end
      %
      % finaly the animation:
      %
      id0= myexp_id('curr');
      for i= anim_list
         if isempty(MYEXPD{i})
            continue;
         end
         figure(201)
         myexp('set', i);
         options2= options;
         options2.sid= sid;
         myexp('sh', options2)
         drawnow
         pause(0.5)
      end
      myexp('set', id0); % recover the original id
      
   case 'phasor_set'
      % user fills the phasor values
      
   case 'phasor_fit'
      % automatically fit phasors to the PSPICE results
      
   case {'save_all', 'save'}
      % save global vars MYEXP (UI info) and MYEXPD (results)
      fname= 'myexp_sav.mat';
      if isfield(options, 'fname')
         fname= options.fname;
      end
      save(fname, 'MYEXP', 'MYEXPD')
      fprintf(1, '-- saved all data to file: %s\n', fname);
      
   case {'load_all'}
      % load previously saved experiments (global vars: MYEXP MYEXPD)
      fname= 'myexp_sav.mat';
      if isfield(options, 'fname')
         fname= options.fname;
      end
      x= load(fname);
      MYEXP= x.MYEXP;
      MYEXPD= x.MYEXPD;
      
   otherwise
      fprintf(1, '** ERROR: invalid command: %s\n', cmd);
end


% -----------------------------------------------------------------
function id= myexp_id(cmd, options)

global MYEXP MYEXPD

switch cmd
   case 'curr'
      % do nothing (the default returns the id)
      
   case 'new'
      % add a new experiment
      MYEXPD{end+1}= [];
      MYEXP.id= length(MYEXPD);
      
   case 'set'
      % select experiment N
      if ~isfield(options, 'arg')
         error('please indicate the exp id');
      end
      id= round(options.arg);
      if id<1 || length(MYEXPD)<id
         error(['experiment id=' num2str(id) ' is not in [1..max]'])
      end
      MYEXP.id= id;
      
   case 'next'
      myexp_id('set', struct('arg', MYEXP.id+1) );
      
   case 'prev'
      myexp_id('set', struct('arg', MYEXP.id-1) );
end

id= MYEXP.id; % the return data
return


% -----------------------------------------------------------------
function sim_config_data_write(freqHz)
fo= fopen('dbode.stl', 'wt');
if fo<1
   disp('** Failed fopen of dbode.stl to write')
end
fprintf(fo, '* dbode.stl written on Mon May 25 09:00:00 2009\n');
fprintf(fo, '* by Stimulus Editor -- Evaluation Version 9.1\n');
fprintf(fo, ';!Stimulus Get\n');
fprintf(fo, ';! my_vstim Analog\n');
fprintf(fo, ';!Ok\n');
fprintf(fo, ';!Plot Axis_Settings\n');
fprintf(fo, ';!Xrange 0s 3ms\n');
fprintf(fo, ';!Yrange -5 5\n');
fprintf(fo, ';!AutoUniverse\n');
fprintf(fo, ';!XminRes 1ns\n');
fprintf(fo, ';!YminRes 1n\n');
fprintf(fo, ';!Ok\n');
fprintf(fo, '.STIMULUS my_vstim SIN( 0V 1V %fHz 0 0 0 )\n', freqHz);
fclose(fo);


function ret= sim_config_data_read
fid= fopen('dbode.stl');
while 1
   tline = fgetl(fid);
   if ~ischar(tline), break, end
   %disp(tline)
   if isempty(strfind(tline, '.STIMULUS'))
      continue;
   end

   % found the desired line, now get the params
   %   find the sin string area
   %   remove all the non-digit (keep spaces)
   %
   ind= strfind(lower(tline), 'sin(');
   str= tline(ind(1):end);
   [tmp, ind]= find(('0'<=str & str<='9') | (str=='.') | (str==' '));
   str= str(ind);
   ret= str2num(str);
end
fclose(fid);


% -----------------------------------------------------------------
function show_myexp_info(id, options)
global MYEXP MYEXPD

if isfield(options, 'min_info')
   if isempty(MYEXPD{id})
      fprintf(1, '  Exp%d is empty.\n', id);
   else
      str= sinfo2str(get_sinfo(id));
      fprintf(1, '  Exp%d %s\n', id, str);
   end
   return % do nothing more
end

if isempty(MYEXPD{id})
   fprintf(1, '** Experiment %d is empty.\n', id);
else
   fprintf(1, '-- Experiment %d:\n', id);
   str= sinfo2str(get_sinfo(id));
   fprintf(1, '   Input signal %s\n', str);
   fprintf(1, '   Simulation data loaded:\n', str);
   disp(MYEXPD{id})
   %MYEXPD{id}.Name
   if isfield(options, 'names')
      names= MYEXPD{id}.Name;
      for i=1:length(names)
         fprintf(1, ' %s', names{i});
      end
      fprintf(1,'\n');
   end
end
return


% -----------------------------------------------------------------
function sinfo= get_sinfo(id)
global MYEXP MYEXPD
sinfo.simDC= MYEXP.simDC(id);
sinfo.simAmpl= MYEXP.simAmpl(id);
sinfo.simFreqHz= MYEXP.simFreqHz(id);


function str= sinfo2str(sinfo)
str= sprintf('Vin(t)= %.1f sin( %f t ) + %.1f [V]  (f=%.3f[Hz])', ...
   sinfo.simAmpl, sinfo.simFreqHz*2*pi, sinfo.simDC, ...
   sinfo.simFreqHz);


function show_signal(id, options)
global MYEXP MYEXPD
if isempty(MYEXPD{id})
   fprintf(1, '** Experiment %d is empty.\n', id);
else
   fprintf(1, '-- Experiment %d:\n', id);
   %MYEXPD{id}
   % signals to show
   sinfo= get_sinfo(id);
   show_signal2(MYEXPD{id}, sinfo, options)
end
return


function [t, y, s2]= get_signals(myexp)
%
% search for special strings
% V(Vin), V(Vout1), ..., V(Vout6)
%
strList= myexp.Name;
s2= {'V(Vin)', 'V(Vout1)', 'V(Vout2)', 'V(Vout3)', 'V(Vout4)', 'V(Vout5)', 'V(Vout6)'};
for i=1:length(s2)
   s2ind(i)= 0;
   for j=1:length(strList)
      if strcmp(strList{j}, s2{i})
         s2ind(i)= j;
         break;
      end
   end
   if s2ind(i)==0,
      s2ind(i)=1;
      warning(['** Signal ' s2{i} ' not found.'])
   end
end

% return the relevant data
t= myexp.Time;
y= myexp.Data(:,s2ind);
return


function show_signal2(myexp, sinfo, options)

% get the relevant data
[t, y, s2]= get_signals(myexp);

sid= 0:6; % default signal Id
if isfield(options, 'sid')
   sid= options.sid;
end
sid= sid+1;
ind= find(sid>7); sid(ind)=7;
ind= find(sid<1); sid(ind)=1;

range= 1:length(t);
if isfield(options, 'lastc') || isfield(options, 'l5c') % show just the last 5 cycles
   %dt= 5*1/sinfo.simFreqHz;
   %tini= t(end)-dt;
   %[tmp, ind]= min(abs(t-tini));
   % how many cycles?
   n2del= 5;
   if isfield(options, 'lastc')
      n2del= options.lastc;
   end
   n= floor(t(end)/(1/sinfo.simFreqHz));
   tini= (n-n2del)*(1/sinfo.simFreqHz);
   [tmp, ind]= min(abs(t-tini));
   range= ind(1):length(t);
end

titleStr= sinfo2str(sinfo);

if isfield(options, 'xyplot')
   %
   % xy plot, i.e. y(x), time is ignored
   %
   if length(sid)<2
      error('requiring at least two signals but length(sid)<2');
   end
   plot(y(range,sid(1)), y(range,sid(2)))
   title(titleStr);
   xlabel(s2{sid(1)})
   ylabel(s2{sid(2)})
   grid on
else
   %
   % normal time plot, i.e. y(t)
   %
   plot(t(range), y(range,sid), '.-')
   title(titleStr);
   xlabel('time [sec]')
   ylabel('voltage [V]')
   str= 'legend(';
   for i=1:length(sid)
      if i>1, str= [str ',']; end
      str= [str '''' s2{sid(i)} ''''];
   end
   str= [str ');'];
   eval(str)
   grid on
end
