function [FSM, curr_state]= TM_ini( machineId, options )
%
% Define a Finite State Machine (FSM) for a Turing Machine

% Oct2016, J. Gaspar

if nargin<2, options= []; end
if nargin<1, [FSM, curr_state]= TM_ini( 'BusyBeaver3' ); return; end

switch machineId
    case 'BusyBeaver3', FSM= def_BusyBeaver3;
    case 'BusyBeaver4', FSM= def_BusyBeaver4;
    case 'BusyBeaver5', FSM= def_BusyBeaver5;
    otherwise
        error('inv machineId');
end

curr_state= FSM{1,1};

return; % end of function


% -----------------------------------------------------------------------
function FSM= replace_quotation_marks( FSM )
for i=1:size(FSM,1)
    for j=1:size(FSM,2)
        FSM{i,j}= strrep(FSM{i,j}, '"', '''');
    end
end


function FSM= convert_table_to_list( tbl )
%
% tbl is a list of strings, each one encoding actions and transitions for
% one state
%
% FSM has four columns: curr_state, true_false_condition, actions, next_state

FSM= {};
for i=1:length(tbl)
    str= tbl{i};
    FSM{i,1}= str(1); % current state
    FSM{i,2}= ['TM_tape("read")==' str(2)]; % condition
    FSM{i,3}= ['TM_tape("write",' str(3) '); TM_tape("right");'];
    if str(4)=='L'
        FSM{i,3}= strrep(FSM{i,3}, '"right"', '"left"');
    elseif str(4)=='N'
        FSM{i,3}= strrep(FSM{i,3}, '"right"', '"null_move"');
    end
    FSM{i,4}= str(5); % next state
    if str(5)=='H'
        FSM{i,4}= 'halt';
    end
end
FSM= replace_quotation_marks( FSM );


% -----------------------------------------------------------------------
function FSM= def_BusyBeaver3_v0
%
% FSM has four columns: curr_state, true_false_condition, actions, next_state
%
%          |- T/F cond -----|    |- write action and move action ----|
FSM= {
    'A',  'TM_tape("read")==0',  'TM_tape("write",1); TM_tape("right");',  'B';
    'A',  'TM_tape("read")==1',  'TM_tape("write",1); TM_tape("left");' ,  'C';
    'B',  'TM_tape("read")==0',  'TM_tape("write",1); TM_tape("left");' ,  'A';
    'B',  'TM_tape("read")==1',  'TM_tape("write",1); TM_tape("right");',  'B';
    'C',  'TM_tape("read")==0',  'TM_tape("write",1); TM_tape("left");' ,  'B';
    'C',  'TM_tape("read")==1',  'TM_tape("write",1); TM_tape("null_move");', 'halt';
    };

FSM= replace_quotation_marks( FSM );


function FSM= def_BusyBeaver3_v1

tbl= {'A01RB', 'A11LC', 'B01LA', 'B11RB', 'C01LB', 'C11NH'};
FSM= convert_table_to_list( tbl );


% -----------------------------------------------------------------------
function FSM= def_BusyBeaver3
% see https://en.wikipedia.org/wiki/Busy_beaver
%  3-state, 2-symbol busy beaver
%
tbl= {'A01RB', 'A11RH', 'B00RC', 'B11RB', 'C01LC', 'C11LA'};
FSM= convert_table_to_list( tbl );


function FSM= def_BusyBeaver4
% see https://en.wikipedia.org/wiki/Busy_beaver
%  4-state, 2-symbol busy beaver
%
tbl= {'A01RB', 'A11LB', 'B01LA', 'B10LC', 'C01RH', 'C11LD', 'D01RD', 'D10RA'};
% In order to mirror the tape usage, you can simply exchange L<->R :
% tbl= {'A01LB', 'A11RB', 'B01RA', 'B10RC', 'C01LH', 'C11RD', 'D01LD', 'D10LA'};
FSM= convert_table_to_list( tbl );


function FSM= def_BusyBeaver5
% see https://en.wikipedia.org/wiki/Busy_beaver
%  current 5-state, 2-symbol best contender (possible busy beaver)
%
tbl= {'A01RB', 'A11LC', 'B01RC', 'B11RB', 'C01RD', 'C10LE', 'D01LA', 'D11LD', 'E01RH', 'E10LA'};
FSM= convert_table_to_list( tbl );
